#!/bin/sh

# Mailing list handler.
# Usage:
#     listserv list-command
# Arguments:
#     list      - name of the mailing list,
#     command   - one of commands: add or subscribe,
#                 delete or unsubscribe, check or show, info or help.
# Commands:
#     add       - add user address to the mailing list,
#     delete    - delete user from the mailing list,
#     check     - get list of all subscribed users,
#     info      - get information about mailing list.
# Author:
#     Serge Vakulenko, vak@kiae.su

# Information about mailing lists is kept in directory /usr/lib/mail/lists.
# This directory should be writable by sendmail or whoever calls listserv
# (probably user 'daemon').

cmd=$1
list=`echo $cmd | sed 's/^\(.*\)-[^-]*$/\1/'`
listfile=/etc/mail/lists/$list.list
infofile=/etc/mail/lists/$list.info
permfile=/etc/mail/lists/$list.perm
infoserv=/etc/mail/listserv.info
mail=/usr/sbin/sendmail
host=`hostname`
tmp=/tmp/lsv$$

# Extract address of user from the mail header.
# Use header fields "Reply-To", "Resent-From", "From", "Sender".
# Never reply to "uucp@" and "MAILER-DAEMON".
user=`awk -F: 'BEGIN {sender="null"}
	/^[rR][eE][pP][lL][yY]-[tT][oO] *:/ {replyto=$2}
	/^[rR][eE][sS][eE][nN][tT]-[fF][rR][oO][mM] *:/ {resentfrom=$2}
	/^[fF][rR][oO][mM] *:/ {from=$2}
	/^[sS][eE][nN][dD][eE][rR] *:/ {sender=$2}
	/^$/ {exit}
	END {
		sender = replyto?replyto:resentfrom?resentfrom:from?from:sender
		if (sender ~ /uucp@/) sender = ""
		if (sender ~ /MAILER-DAEMON/) sender = ""
		print sender
	}'`
user=`echo $user | sed 's/^.*<\(.*\)>.*$/\1/
	s/^ *\([^( ]*\) *(.*$/\1/'`
if [ x$user = x ]; then
	exit 0
fi

trap 'rm -f $tmp' 0 1 2 3 15
if [ x$cmd = xinfo -o x$cmd = xhelp ]; then
	cat << ENDOFHELP
Mailing List Server, Copyright (C) 1993 Serge Vakulenko, vak@kiae.su

This is the mailing list server, an automated program,
which allows you to get information about mailing lists,
available at $host, subscribe to them, and unsubscribe at will.

Mailing lists are created and deleted by human, the list server
administrator.  Call him if you are sure there is the reason
to create a new mailing list or delete an old one.

All actions with list server are performed by sending the letter
to domain <listserv.$host>.  The body of the message
is ignored, and the action is determined by the user part of address.

For example, to get this help, send an empty letter to address:

	help@listserv.$host

To subscribe to, assume, test-list mailing list, e-mail to:

	test-list-subscribe@listserv.$host

To unsubscribe from test-list:

	test-list-unsubscribe@listserv.$host

To see all address in the list test-list:

	test-list-show@listserv.$host

To get information about test-list:

	test-list-info@listserv.$host

To send letter to the mailing list (to subscribers, not to the list server!),
send mail to:

	test-list@$host

All responses from list server have return address with "-owner" suffix.
Don't send letters to these addresses, they are commonly directed to /dev/null.

The following mailing lists are available:

ENDOFHELP
	for i in /usr/lib/mail/lists/*.list; do
		list=`basename $i .list`
		echo "        $list"
		infofile=/usr/lib/mail/lists/$list.info
		if [ -f $infofile ]; then
			sed 's/^/                /' $infofile
		else
			echo "                No information"
		fi
	done
	subj="[Listserv] Information about Listserv"
	list=listserv
elif [ -f $listfile ]; then
	case $cmd in
	*-add|*-subscribe)
		if [ -f $permfile ] && [ x`echo $user | egrep -f $permfile` != x$user ]; then
			echo "$user is not permitted to subscribe to the mailing list $list"
			subj="[Listserv] Permission denied"
		elif grep -s "$user" $listfile; then
			echo "$user is already subscribed to the mailing list $list"
			subj="[Listserv] You already subscribed to $list"
		else
			grep -v "$user" $listfile > $listfile~
			if echo "$user" >> $listfile~ && mv $listfile~ $listfile; then
				echo "$user subscribed to the mailing list $list"
				subj="[Listserv] You subscribed to $list"
			else
				echo "Error subscribing $user to the mailing list $list"
				subj="[Listserv] Cannot subscribe"
			fi
		fi
		;;
	*-delete|*-unsubscribe)
		if [ -f $permfile ] && [ x`echo $user | egrep -f $permfile` != x$user ]; then
			echo "$user is not permitted to unsubscribe from the mailing list $list"
			subj="[Listserv] Permission denied"
		else
			grep -v "$user" $listfile > $listfile~
			mv $listfile~ $listfile
			if [ $? = 0 ]; then
				echo "$user unsubscribed from the mailing list $list"
				subj="[Listserv] You unsubscribed from $list"
			else
				echo "Error unsubscribing $user from the mailing list $list"
				subj="[Listserv] Cannot unsubscribe"
			fi
		fi
		;;
	*-check|*-show)
		echo "Mailing list $list = {"
		sed 's/^/    /' $listfile
		echo "}"
		subj="[Listserv] Contents of list $list"
		;;
	*-info|*-help)
		if [ -f $infofile ]; then
			cat $infofile
		else
			echo "No information about mailing list $list"
		fi
		subj="[Listserv] Information about $list"
		;;
	*)
		echo "No such command: $cmd"
		subj="[Listserv] Bad command"
		;;
	esac
else
	echo "No such mailing list: $list"
	subj="[Listserv] No such list $list"
	list=listserv
fi > $tmp

(
	echo "From: $list-owner"
	echo "Subject: $subj"
	echo "To: $user"
	echo
	cat $tmp
) | $mail -r $list-owner $user
